<?php

namespace App\Http\Controllers;

use App\Models\InterviewQuestion;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Exception;

class InterviewQuestionController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(): JsonResponse
    {
        try {
            $questions = InterviewQuestion::with('job')->get();
            return response()->json([
                'success' => true,
                'data' => $questions,
                'message' => 'Questions retrieved successfully'
            ], 200);
        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve questions: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request): JsonResponse
    {
        try {
            $validated = $request->validate([
                'Job_id' => 'required|exists:jobs,Job_id',
                'Question_text' => 'required|string|max:65535',
                'Expected_answer' => 'nullable|string|max:65535',
                'Difficulty_level' => 'nullable|in:easy,medium,hard',
                'Time_limit' => 'nullable|integer|min:1',
            ]);

            $question = InterviewQuestion::create($validated);
            $question->load('job');

            return response()->json([
                'success' => true,
                'data' => $question,
                'message' => 'Question created successfully'
            ], 201);
        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to create question: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show($id): JsonResponse
    {
        try {
            $question = InterviewQuestion::with('job')->findOrFail($id);
            return response()->json([
                'success' => true,
                'data' => $question,
                'message' => 'Question retrieved successfully'
            ], 200);
        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Question not found or error: ' . $e->getMessage()
            ], 404);
        }
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id): JsonResponse
    {
        try {
            $question = InterviewQuestion::findOrFail($id);
            $validated = $request->validate([
                'Job_id' => 'required|exists:jobs,Job_id',
                'Question_text' => 'required|string|max:65535',
                'Expected_answer' => 'nullable|string|max:65535',
                'Difficulty_level' => 'nullable|in:easy,medium,hard',
                'Time_limit' => 'nullable|integer|min:1',
            ]);

            $question->update($validated);
            $question->load('job');

            return response()->json([
                'success' => true,
                'data' => $question,
                'message' => 'Question updated successfully'
            ], 200);
        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update question: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id): JsonResponse
    {
        try {
            $question = InterviewQuestion::findOrFail($id);
            $question->delete();

            return response()->json([
                'success' => true,
                'data' => null,
                'message' => 'Question deleted successfully'
            ], 204);
        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete question: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get questions by Job ID.
     */
    public function getByJobId($jobId): JsonResponse
    {
        try {
            $questions = InterviewQuestion::where('Job_id', $jobId)->with('job')->get();
            if ($questions->isEmpty()) {
                return response()->json([
                    'success' => false,
                    'message' => 'No questions found for Job ID ' . $jobId
                ], 404);
            }

            return response()->json([
                'success' => true,
                'data' => $questions,
                'message' => 'Questions retrieved successfully for Job ID ' . $jobId
            ], 200);
        } catch (Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve questions: ' . $e->getMessage()
            ], 500);
        }
    }
}
